// Copyright (C) 2019  Allen Li
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Command binpack is a Go generator for packing static assets into Go
// packages.
//
// For example, if you have a file base.html in your package containing:
//
//   <html><body>{{.body}}</body></html>
//
// You can add a generate line anywhere in your package:
//
//   //go:generate binpack -name myTemplate base.html
//
// Then running go generate will produce a myTemplate_packed.go containing:
//
//   // Code generated by "binpack -name myTemplate base.html"; DO NOT EDIT.
//
//   package foo
//
//   const myTemplate = "<html><body>{{.body}}</body></html>"
//
// This also works for binary files, although the generated source is
// very inefficient as it is encoded as a string literal (but after
// compiling the data should not take up any extra space).
package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/build"
	"io/ioutil"
	"log"
	"os"
	"strings"
)

var (
	name = flag.String("name", "", "name of constant to store file data")
)

func Usage() {
	fmt.Fprintf(os.Stderr, "Usage:\n")
	fmt.Fprintf(os.Stderr, "\tbinpack [flags] -name name file\n")
	flag.PrintDefaults()
}

func main() {
	log.SetFlags(0)
	log.SetPrefix("binpack: ")
	flag.Usage = Usage
	flag.Parse()
	if *name == "" || flag.NArg() != 1 {
		flag.Usage()
		os.Exit(2)
	}
	inputFile := flag.Arg(0)
	data, err := ioutil.ReadFile(inputFile)
	if err != nil {
		log.Fatalf("reading file: %s", err)
	}

	var b bytes.Buffer
	fmt.Fprintf(&b, "// Code generated by \"binpack %s\"; DO NOT EDIT.\n",
		strings.Join(os.Args[1:], " "))
	fmt.Fprintf(&b, "\n")
	fmt.Fprintf(&b, "package %s\n", getPackageName())
	fmt.Fprintf(&b, "\n")
	fmt.Fprintf(&b, "const %s = %q\n", *name, data)

	output := fmt.Sprintf("%s_packed.go", *name)
	if err := ioutil.WriteFile(output, b.Bytes(), 0644); err != nil {
		log.Fatalf("writing output: %s", err)
	}
}

func getPackageName() string {
	dir, err := os.Getwd()
	if err != nil {
		log.Fatalf("cannot get current directory: %s", err)
	}
	pkg, err := build.Default.ImportDir(dir, 0)
	if err != nil {
		log.Fatalf("cannot process directory: %s", err)
	}
	return pkg.Name
}
